{ This unit defines the structure of the model. There are four functions. The
  first function, called counts, defines the number, names, and units of the
  model, the state variables, the process variables, the driver variables and
  the parameters. The second function, called processes, is the actual equations
  which make up the model. The third function, derivs, calculates the
  derivatives of state variables. And the fourth function, parcount, is used to
  automatically number the parameters consecutively. 
    The state variables, driver variables, process variables and parameters are
  all stored in global arrays, called stat, drive, proc, and par, respectively.
  The function counts accesses the global arrays directly but the other functions
  operate on copies of the global arrays. }
unit equations;

interface

uses  stypes, math;

PROCEDURE counts;
PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
function ParCount(processnum:integer) : integer;

var
  tproc:processarray;
  tstat:statearray;
  sensflag:boolean;
  newyear:Boolean = false;
  DayofYear: double = 0;

implementation

uses frontend, calculate, options;

           { Do not make modifcations above this line. }
{*****************************************************************************}

{ This procedure defines the model. The number of parameters, state, driver and
  process variables are all set in this procedure. The model name, version
  number and time unit are also set here. This procedure accesses the global
  arrays containing the the parameters, state, driver and process variables and
  the global structure ModelDef directly, to save memory space. }
PROCEDURE counts;
var
 i,npar,CurrentProc:integer;
begin
{ Set the modelname, version and time unit. }
ModelDef.modelname := 'GTHM Daily 4 layers';
ModelDef.versionnumber := '1.0';
ModelDef.timeunit := 'days';
ModelDef.contactperson := 'Feifei/Marc/Bonnie';
ModelDef.contactaddress1 := 'GT';
ModelDef.contactaddress2 := '';
ModelDef.contactaddress3 := 'Atlanta, GA';

{ Set the number of state variables in the model. The maximum number of state
  variables is maxstate, in unit stypes. }
ModelDef.numstate := 5;

{ Enter the name, units and symbol for each state variable. The maximum length
  of the state variable name is 17 characters and the maximum length for units
  and symbol is stringlength (specified in unit stypes) characters. }
 
 
with stat[1] do
 begin
    name:='*Soil Water1';  units:='mm H2O';  symbol:='Ws[1]'; 
 end;
 
with stat[2] do
 begin
    name:='*Soil Water2';  units:='mm H2O';  symbol:='Ws[2]'; 
 end;
 
with stat[3] do
 begin
    name:='*Soil Water3';  units:='mm H2O';  symbol:='Ws[3]'; 
 end;
 
with stat[4] do
 begin
    name:='*Soil Water4';  units:='mm H2O';  symbol:='Ws[4]'; 
 end;
 
with stat[5] do
 begin
    name:='Standing water';  units:='mm H2O';  symbol:='Wst';
 end;

{ Set the total number of processes in the model. The first numstate processes
  are the derivatives of the state variables. The maximum number of processes is
  maxparam, in unit stypes. }
ModelDef.numprocess := ModelDef.numstate + 101;

{ For each process, set proc[i].parameters equal to the number of parameters
  associated with that process, and set IsDiscrete to true or false. After each
  process, set the name, units, and symbol for all parameters associated with
  that process. Note that Parcount returns the total number of parameters in
  all previous processes. }
 
CurrentProc := ModelDef.numstate + 1;
With proc[CurrentProc] do
   begin
      name       := '*Soil Properties layers1';
      units       := 'none';
      symbol       := 'dum[1]';
      parameters       := 16;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Porosity1';  units:='fraction';  symbol:='phi[1]';
 end;
with par[npar + 2] do
 begin
    name:='Layer thickness1';  units:='m';  symbol:='d[1]';
 end;
with par[npar + 3] do
 begin
    name:='H2O at wilting1';  units:='mm H2O';  symbol:='thetaw[1]';
 end;
with par[npar + 4] do
 begin
    name:='H2O at field capacity1';  units:='mm H2O';  symbol:='thetafc[1]';
 end;
with par[npar + 5] do
 begin
    name:='Porosity2';  units:='fraction';  symbol:='phi[2]';
 end;
with par[npar + 6] do
 begin
    name:='Layer thickness2';  units:='m';  symbol:='d[2]';
 end;
with par[npar + 7] do
 begin
    name:='H2O at wilting2';  units:='mm H2O';  symbol:='thetaw[2]';
 end;
with par[npar + 8] do
 begin
    name:='H2O at field capacity2';  units:='mm H2O';  symbol:='thetafc[2]';
 end;
with par[npar + 9] do
 begin
    name:='Porosity3';  units:='fraction';  symbol:='phi[3]';
 end;
with par[npar + 10] do
 begin
    name:='Layer thickness3';  units:='m';  symbol:='d[3]';
 end;
with par[npar + 11] do
 begin
    name:='H2O at wilting3';  units:='mm H2O';  symbol:='thetaw[3]';
 end;
with par[npar + 12] do
 begin
    name:='H2O at field capacity3';  units:='mm H2O';  symbol:='thetafc[3]';
 end;
with par[npar + 13] do
 begin
    name:='Porosity4';  units:='fraction';  symbol:='phi[4]';
 end;
with par[npar + 14] do
 begin
    name:='Layer thickness4';  units:='m';  symbol:='d[4]';
 end;
with par[npar + 15] do
 begin
    name:='H2O at wilting4';  units:='mm H2O';  symbol:='thetaw[4]';
 end;
with par[npar + 16] do
 begin
    name:='H2O at field capacity4';  units:='mm H2O';  symbol:='thetafc[4]';
 end;
 
CurrentProc := ModelDef.numstate + 2;
With proc[CurrentProc] do
   begin
      name       := '*Soil Properties layers2';
      units       := 'none';
      symbol       := 'dum[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 3;
With proc[CurrentProc] do
   begin
      name       := '*Soil Properties layers3';
      units       := 'none';
      symbol       := 'dum[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 4;
With proc[CurrentProc] do
   begin
      name       := '*Soil Properties layers4';
      units       := 'none';
      symbol       := 'dum[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 5;
With proc[CurrentProc] do
   begin
      name       := 'Soil Properties grid';
      units       := 'none';
      symbol       := 'dum2';
      parameters       := 6;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Number of active layers';  units:='unitless';  symbol:='numactivelayer';
 end;
with par[npar + 2] do
 begin
    name:='Cell size';  units:='m';  symbol:='dd';
 end;
with par[npar + 3] do
 begin
    name:='Surface Lat Sat Hyd Cond';  units:='mm H2O day-1';  symbol:='Ksl0';
 end;
with par[npar + 4] do
 begin
    name:='Surface Ver Sat Hyd Cond';  units:='mm H2O day-1';  symbol:='Ksv0';
 end;
with par[npar + 5] do
 begin
    name:='Lateral decay constant';  units:='??';  symbol:='fl';
 end;
with par[npar + 6] do
 begin
    name:='Vertical decay constant';  units:='??';  symbol:='fv';
 end;
 
CurrentProc := ModelDef.numstate + 6;
With proc[CurrentProc] do
   begin
      name       := '*Evapotranspiration1';
      units       := 'mm H2O day-1';
      symbol       := 'Et[1]';
      parameters       := 4;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Root fraction in layer1';  units:='fraction';  symbol:='r[1]';
 end;
with par[npar + 2] do
 begin
    name:='Root fraction in layer2';  units:='fraction';  symbol:='r[2]';
 end;
with par[npar + 3] do
 begin
    name:='Root fraction in layer3';  units:='fraction';  symbol:='r[3]';
 end;
with par[npar + 4] do
 begin
    name:='Root fraction in layer4';  units:='fraction';  symbol:='r[4]';
 end;
 
CurrentProc := ModelDef.numstate + 7;
With proc[CurrentProc] do
   begin
      name       := '*Evapotranspiration2';
      units       := 'mm H2O day-1';
      symbol       := 'Et[2]';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 8;
With proc[CurrentProc] do
   begin
      name       := '*Evapotranspiration3';
      units       := 'mm H2O day-1';
      symbol       := 'Et[3]';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 9;
With proc[CurrentProc] do
   begin
      name       := '*Evapotranspiration4';
      units       := 'mm H2O day-1';
      symbol       := 'Et[4]';
      parameters       := 0;
      ptype       := ptGroup2;
   end;
 
CurrentProc := ModelDef.numstate + 10;
With proc[CurrentProc] do
   begin
      name       := 'Potential Et';
      units       := 'mm H2O day-1';
      symbol       := 'Pet';
      parameters       := 2;
      ptype       := ptGroup2;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Density of air';  units:='g m-3';  symbol:='rhoair';
 end;
with par[npar + 2] do
 begin
    name:='Decay constant';  units:='unitless';  symbol:='bet';
 end;
 
CurrentProc := ModelDef.numstate + 11;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 1-';
      units       := 'unitless';
      symbol       := 'SlH1';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 12;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 2-';
      units       := 'unitless';
      symbol       := 'SlH2';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 13;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 3-';
      units       := 'unitless';
      symbol       := 'SlH3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 14;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 4-';
      units       := 'unitless';
      symbol       := 'SlH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 15;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 5-';
      units       := 'unitless';
      symbol       := 'SlH5';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 16;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 6-';
      units       := 'unitless';
      symbol       := 'SlH6';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 17;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 7-';
      units       := 'unitless';
      symbol       := 'SlH7';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 18;
With proc[CurrentProc] do
   begin
      name       := 'Slope-Heading 8-';
      units       := 'unitless';
      symbol       := 'SlH8';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 19;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 1-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH1[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 20;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 1-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH1[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 21;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 1-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH1[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 22;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 1-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH1[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 23;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 2-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH2[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 24;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 2-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH2[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 25;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 2-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH2[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 26;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 2-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH2[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 27;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 3-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH3[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 28;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 3-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH3[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 29;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 3-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH3[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 30;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 3-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH3[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 31;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 4-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH4[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 32;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 4-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH4[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 33;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 4-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH4[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 34;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 4-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH4[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 35;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 5-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH5[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 36;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 5-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH5[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 37;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 5-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH5[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 38;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 5-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH5[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 39;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 6-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH6[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 40;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 6-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH6[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 41;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 6-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH6[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 42;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 6-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH6[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 43;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 7-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH7[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 44;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 7-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH7[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 45;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 7-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH7[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 46;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 7-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH7[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 47;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 8-1';
      units       := 'mm H2O day-1';
      symbol       := 'QlH8[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 48;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 8-2';
      units       := 'mm H2O day-1';
      symbol       := 'QlH8[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 49;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 8-3';
      units       := 'mm H2O day-1';
      symbol       := 'QlH8[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 50;
With proc[CurrentProc] do
   begin
      name       := '*Runoff lateral-Head 8-4';
      units       := 'mm H2O day-1';
      symbol       := 'QlH8[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 51;
With proc[CurrentProc] do
   begin
      name       := '*Lateral logistic curve1';
      units       := 'unitless';
      symbol       := 'fq[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 52;
With proc[CurrentProc] do
   begin
      name       := '*Lateral logistic curve2';
      units       := 'unitless';
      symbol       := 'fq[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 53;
With proc[CurrentProc] do
   begin
      name       := '*Lateral logistic curve3';
      units       := 'unitless';
      symbol       := 'fq[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 54;
With proc[CurrentProc] do
   begin
      name       := '*Lateral logistic curve4';
      units       := 'unitless';
      symbol       := 'fq[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 55;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Lat1';
      units       := '??';
      symbol       := 'Ksl[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 56;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Lat2';
      units       := '??';
      symbol       := 'Ksl[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 57;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Lat3';
      units       := '??';
      symbol       := 'Ksl[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 58;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Lat4';
      units       := '??';
      symbol       := 'Ksl[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 59;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 1-1';
      units       := '??';
      symbol       := 'bl1[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 60;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 1-2';
      units       := '??';
      symbol       := 'bl1[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 61;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 1-3';
      units       := '??';
      symbol       := 'bl1[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 62;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 1-4';
      units       := '??';
      symbol       := 'bl1[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 63;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 2-1';
      units       := '??';
      symbol       := 'bl2[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 64;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 2-2';
      units       := '??';
      symbol       := 'bl2[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 65;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 2-3';
      units       := '??';
      symbol       := 'bl2[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 66;
With proc[CurrentProc] do
   begin
      name       := '*Runoff parameter 2-4';
      units       := '??';
      symbol       := 'bl2[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 67;
With proc[CurrentProc] do
   begin
      name       := '*Drainage from layer1';
      units       := 'mm H2O day-1';
      symbol       := 'G[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 68;
With proc[CurrentProc] do
   begin
      name       := '*Drainage from layer2';
      units       := 'mm H2O day-1';
      symbol       := 'G[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 69;
With proc[CurrentProc] do
   begin
      name       := '*Drainage from layer3';
      units       := 'mm H2O day-1';
      symbol       := 'G[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 70;
With proc[CurrentProc] do
   begin
      name       := '*Drainage from layer4';
      units       := 'mm H2O day-1';
      symbol       := 'G[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 71;
With proc[CurrentProc] do
   begin
      name       := '*Vertical logistic curv1';
      units       := 'unitless';
      symbol       := 'fg[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 72;
With proc[CurrentProc] do
   begin
      name       := '*Vertical logistic curv2';
      units       := 'unitless';
      symbol       := 'fg[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 73;
With proc[CurrentProc] do
   begin
      name       := '*Vertical logistic curv3';
      units       := 'unitless';
      symbol       := 'fg[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 74;
With proc[CurrentProc] do
   begin
      name       := '*Vertical logistic curv4';
      units       := 'unitless';
      symbol       := 'fg[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 75;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Vert1';
      units       := '??';
      symbol       := 'Ksv[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 76;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Vert2';
      units       := '??';
      symbol       := 'Ksv[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 77;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Vert3';
      units       := '??';
      symbol       := 'Ksv[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 78;
With proc[CurrentProc] do
   begin
      name       := '*Hydraul Cond Vert4';
      units       := '??';
      symbol       := 'Ksv[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 79;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 1-1';
      units       := '??';
      symbol       := 'bv1[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 80;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 1-2';
      units       := '??';
      symbol       := 'bv1[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 81;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 1-3';
      units       := '??';
      symbol       := 'bv1[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 82;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 1-4';
      units       := '??';
      symbol       := 'bv1[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 83;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 2-1';
      units       := '??';
      symbol       := 'bv2[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 84;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 2-2';
      units       := '??';
      symbol       := 'bv2[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 85;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 2-3';
      units       := '??';
      symbol       := 'bv2[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 86;
With proc[CurrentProc] do
   begin
      name       := '*Drainage parameter 2-4';
      units       := '??';
      symbol       := 'bv2[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 87;
With proc[CurrentProc] do
   begin
      name       := 'Infiltration';
      units       := 'mm H2O';
      symbol       := 'Inf';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 88;
With proc[CurrentProc] do
   begin
      name       := '*Excess Water1';
      units       := 'mm H2O';
      symbol       := 'Wsex[1]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 89;
With proc[CurrentProc] do
   begin
      name       := '*Excess Water2';
      units       := 'mm H2O';
      symbol       := 'Wsex[2]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 90;
With proc[CurrentProc] do
   begin
      name       := '*Excess Water3';
      units       := 'mm H2O';
      symbol       := 'Wsex[3]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 91;
With proc[CurrentProc] do
   begin
      name       := '*Excess Water4';
      units       := 'mm H2O';
      symbol       := 'Wsex[4]';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 92;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 1';
      units       := 'mm H2O day-1';
      symbol       := 'QsH1';
      parameters       := 1;
      ptype       := ptGroup1;
   end;
npar:=ParCount(CurrentProc);
with par[npar + 1] do
 begin
    name:='Chezy number';  units:='??';  symbol:='Ch';
 end;
 
CurrentProc := ModelDef.numstate + 93;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 2';
      units       := 'mm H2O day-1';
      symbol       := 'QsH2';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 94;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 3';
      units       := 'mm H2O day-1';
      symbol       := 'QsH3';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 95;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 4';
      units       := 'mm H2O day-1';
      symbol       := 'QsH4';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 96;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 5';
      units       := 'mm H2O day-1';
      symbol       := 'QsH5';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 97;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 6';
      units       := 'mm H2O day-1';
      symbol       := 'QsH6';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 98;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 7';
      units       := 'mm H2O day-1';
      symbol       := 'QsH7';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 99;
With proc[CurrentProc] do
   begin
      name       := 'Surface Runoff-Head 8';
      units       := 'mm H2O day-1';
      symbol       := 'QsH8';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 100;
With proc[CurrentProc] do
   begin
      name       := 'Total Surface Runoff';
      units       := 'mm H2O day-1';
      symbol       := 'QsT';
      parameters       := 0;
      ptype       := ptGroup1;
   end;
 
CurrentProc := ModelDef.numstate + 101;
With proc[CurrentProc] do
   begin
      name       := 'Total Lateral Runoff';
      units       := 'mm H2O day-1';
      symbol       := 'QlT';
      parameters       := 0;
      ptype       := ptGroup1;
   end;

{ Set the total number of drivers in the model. The maximum number of drivers is
  maxdrive, in unit stypes. }
ModelDef.numdrive := 16;

{ Set the names, units, and symbols of the drivers. The maximum length for the
  name, units and symbol is 20 characters. }
 
with drive[1] do
 begin
    name:='D: Daily Temp Max';  units:='deg C';  symbol:='Tamax';
 end;
 
with drive[2] do
 begin
    name:='D: Daily Temp Min';  units:='deg C';  symbol:='Tamin';
 end;
 
with drive[3] do
 begin
    name:='D: Precipitation surfac';  units:='mm H2O day-1';  symbol:='P';
 end;
 
with drive[4] do
 begin
    name:='D: Cell elevation';  units:='m';  symbol:='Hc';
 end;
 
with drive[5] do
 begin
    name:='D: Cell elev - Head 1-';  units:='m';  symbol:='HH1';
 end;
 
with drive[6] do
 begin
    name:='D: Cell elev - Head 2-';  units:='m';  symbol:='HH2';
 end;
 
with drive[7] do
 begin
    name:='D: Cell elev - Head 3-';  units:='m';  symbol:='HH3';
 end;
 
with drive[8] do
 begin
    name:='D: Cell elev - Head 4-';  units:='m';  symbol:='HH4';
 end;
 
with drive[9] do
 begin
    name:='D: Cell elev - Head 5-';  units:='m';  symbol:='HH5';
 end;
 
with drive[10] do
 begin
    name:='D: Cell elev - Head 6-';  units:='m';  symbol:='HH6';
 end;
 
with drive[11] do
 begin
    name:='D: Cell elev - Head 7-';  units:='m';  symbol:='HH7';
 end;
 
with drive[12] do
 begin
    name:='D: Cell elev - Head 8-';  units:='m';  symbol:='HH8';
 end;
 
with drive[13] do
 begin
    name:='*D: Lat H2O input layer1';  units:='mm H2O day-1';  symbol:='Qin[1]';
 end;
 
with drive[14] do
 begin
    name:='*D: Lat H2O input layer2';  units:='mm H2O day-1';  symbol:='Qin[2]';
 end;
 
with drive[15] do
 begin
    name:='*D: Lat H2O input layer3';  units:='mm H2O day-1';  symbol:='Qin[3]';
 end;
 
with drive[16] do
 begin
    name:='*D: Lat H2O input layer4';  units:='mm H2O day-1';  symbol:='Qin[4]';
 end;

{ The first numstate processes are the derivatives of the state variables. The
  code sets the names, units and symbols accordingly.}
for i:= 1 to ModelDef.numstate do proc[i].name:='d'+stat[i].name+'dt';
for i:= 1 to ModelDef.numstate do proc[i].units := stat[i].units + 't-1';
for i:= 1 to ModelDef.numstate do proc[i].symbol := 'd' + stat[i].symbol + 'dt';

{ Code to sum up the total number of parameters in the model. Do not change the
  next few lines. }
ModelDef.numparam := 0;
for i := 1 to ModelDef.NumProcess do
  ModelDef.numparam := ModelDef.numparam + proc[i].parameters;

end; // counts procedure


{ A procedure to calculate the value of all states and processes at the current
  time. This function accesses time, state variables and process variables by
  reference, ie it uses the same array as the calling routine. It does not use
  the global variables time, stat and proc because values calculated during
  integration might later be discarded. It does access the global variables par,
  drive and ModelDef directly because those values are not modified.

  The model equations are written using variable names which correspond to the
  actual name instead of using the global arrays (i.e. SoilWater instead of
  stat[7].value). This makes it necessary to switch all values into local
  variables, do all the calculations and then put everything back into the
  global variables. Lengthy but worth it in terms of readability of the code. }

// Choose either GlobalPs, ArcticPs, or none here so the appropriate Ps model is compiled below.
{$DEFINE none}

PROCEDURE processes(time:double; dtime:double; var tdrive:drivearray;
                       var tpar:paramarray; var tstat:statearray;
                       var tproc:processarray; CalculateDiscrete:Boolean);
{$IFDEF GlobalPs}
const
// Global Ps parameters
 x1 = 11.04;             x2 = 0.03;
 x5 = 0.216;             x6 = 0.6;
 x7 = 3.332;             x8 = 0.004;
 x9 = 1.549;             x10 = 1.156;
 gammastar = 0;          kCO2 = 995.4;  }
{$ENDIF}

// Modify constant above (line above "procedure processes..." line )to specify
// which Ps model and it's constants should be compiled. Choosing a Ps model
// automatically includes the Et and Misc constants (i.e. Gem is assumed).

{$IFDEF ArcticPs}
const
// Arctic Ps parameters
x1 = 0.192;	x2 = 0.125;
x5 = 2.196;	x6 = 50.41;
x7 = 0.161;	x8 = 14.78;
x9 = 1.146;
gammastar = 0.468;	kCO2 = 500.3;
{$ENDIF}

{$IFDEF ArcticPs OR GlobalPs}
//const
// General Et parameters
aE1 = 0.0004;    aE2 = 150;  aE3 = 1.21;   aE4 = 6.11262E5;

// Other constants
cp = 1.012E-9; //specific heat air MJ kg-1 oC-1
sigmaSB = 4.9e-9; //stefan-Boltzmann MJ m-2 day-1 K-4
S0 = 117.5; //solar constant MJ m-2 day-1
bHI1 =0.23;
bHI2 =0.48;
mw = 2.99; //kg h2o MJ-1
alphaMS = 2; //mm oC-1 day-1                                 }
{$ENDIF}

var
{ List the variable names you are going to use here. Generally, this list
  includes all the symbols you defined in the procedure counts above. The order
  in which you list them does not matter. }
{States}
Ws, dWsdt: array[1..4] of double;
Wst, dWstdt :double;

{processes and associated parameters}
dum, phi, d, thetaw, thetafc: array[1..4] of double;
dum2, numactivelayer, dd, Ksl0, Ksv0, fl, fv :double;
Et, r: array[1..4] of double;
Pet, rhoair, bet, 
SlH1, 
SlH2, 
SlH3, 
SlH4, 
SlH5, 
SlH6, 
SlH7, 
SlH8 :double;
QlH1: array[1..4] of double;
QlH2: array[1..4] of double;
QlH3: array[1..4] of double;
QlH4: array[1..4] of double;
QlH5: array[1..4] of double;
QlH6: array[1..4] of double;
QlH7: array[1..4] of double;
QlH8: array[1..4] of double;
fq: array[1..4] of double;
Ksl: array[1..4] of double;
bl1: array[1..4] of double;
bl2: array[1..4] of double;
G: array[1..4] of double;
fg: array[1..4] of double;
Ksv: array[1..4] of double;
bv1: array[1..4] of double;
bv2: array[1..4] of double;
Inf :double;
Wsex: array[1..4] of double;
QsH1, Ch, 
QsH2, 
QsH3, 
QsH4, 
QsH5, 
QsH6, 
QsH7, 
QsH8, 
QsT, 
QlT, 

{drivers}
Tamax, 
Tamin, 
P, 
Hc, 
HH1, 
HH2, 
HH3, 
HH4, 
HH5, 
HH6, 
HH7, 
HH8 :double;
Qin: array[1..4] of double;

{Other double}
Wsm: array[1..4] of double;
Ta, 
rhovsat, 
esat, 
Dl :double;
z: array[1..4] of double;
wfps

:double; {Final double}
{Other integers}
kjll: integer;
kjlbn: array[1..2] of integer;
kjl: integer;
npar, j, jj, kk, ll, tnum:integer;


{ Functions or procedures }

begin
{ Copy the drivers from the global array, drive, into the local variables. }
Tamax := tdrive[1].value;
Tamin := tdrive[2].value;
P := tdrive[3].value;
Hc := tdrive[4].value;
HH1 := tdrive[5].value;
HH2 := tdrive[6].value;
HH3 := tdrive[7].value;
HH4 := tdrive[8].value;
HH5 := tdrive[9].value;
HH6 := tdrive[10].value;
HH7 := tdrive[11].value;
HH8 := tdrive[12].value;
for jj := 1 to 4 do Qin[jj] := tdrive[13 + jj - 1].value;

{ Copy the state variables from the global array into the local variables. }
for jj := 1 to 4 do Ws[jj] := tstat[0 + jj].value;
Wst := tstat[5].value;

{ And now copy the parameters into the local variables. No need to copy the
  processes from the global array into local variables. Process values will be
  calculated by this procedure.

  Copy the parameters for each process separately using the function ParCount
  to keep track of the number of parameters in the preceeding processes.
  npar now contains the number of parameters in the preceding processes.
  copy the value of the first parameter of this process into it's local
  variable }
npar:=ParCount(ModelDef.numstate + 1);
for jj := 1 to 4 do phi[jj] := par[npar + 1 + (jj - 1)*4].value;
for jj := 1 to 4 do d[jj] := par[npar + 2 + (jj - 1)*4].value;
for jj := 1 to 4 do thetaw[jj] := par[npar + 3 + (jj - 1)*4].value;
for jj := 1 to 4 do thetafc[jj] := par[npar + 4 + (jj - 1)*4].value;

npar:=ParCount(ModelDef.numstate + 5);
numactivelayer := par[npar + 1].value;
dd := par[npar + 2].value;
Ksl0 := par[npar + 3].value;
Ksv0 := par[npar + 4].value;
fl := par[npar + 5].value;
fv := par[npar + 6].value;
 
npar:=ParCount(ModelDef.numstate + 6);
for jj := 1 to 4 do r[jj] := par[npar + 1 + (jj - 1)*1].value;
 
npar:=ParCount(ModelDef.numstate + 10);
rhoair := par[npar + 1].value;
bet := par[npar + 2].value;
 
npar:=ParCount(ModelDef.numstate + 92);
Ch := par[npar + 1].value;
 
 
{ Enter the equations to calculate the processes here, using the local variable
  names defined above. }

For j := 1 to 4 do Dum[j] := 0;
Dum2 := 0;

QlT := 0; // Initialized to zero because this is calculated in the for loop.
G[0] := 0; // These lines will cause an error on compile. To fix the error change
z[0] := 0;  {  the declaration of G near line 1427 and z near line 1466 to array[0..4]
    instead of array[1..4]. This will need to be done EVERY time create equations is run 
    because create equations always make the arrays 1 based.  }

Ta := (Tamax + Tamin)/2;

// Calculate soil water content at saturation
for j := 1 to 4 do
  Wsm[j] := 1000*phi[j]*d[j];

// Supersaturation
for j := 4 downto 1 do
  begin
   if j < 4 then Ws[j] := Ws[j] + Wsex[j+1];
    if Ws[j] > Wsm[j] then
     begin
      Wsex[j] := Ws[j] - Wsm[j];
      Ws[j] := Wsm[j];
     end
    else
     Wsex[j] := 0;
  end;

if Wsex[1] > 0 then
 begin
   Wst := Wst + Wsex[1];
   Wsex[1] := 0;
 end;

// Infiltration
if P + Wst/FmOptions.RunOptions.Time_step >= Ksv0 then
  Inf := Ksv0
else
  Inf := P + Wst/FmOptions.RunOptions.Time_step;

// Evapotranspiration
esat := 6.11*exp(17.3*Ta/(Ta + 237.3));
rhovsat := 0.622*rhoair*esat/1000{Pres};
Pet := 0.0138*{Dl*}rhovsat;         //The Dl is in the GTHM documentation but not the code

// Slope to cell neighbors
if HH1 <> -9999 then
 SlH1 := (Hc - HH1)/dd
else
 SlH1 := 0;
if HH2 <> -9999 then
 SlH2 := (Hc - HH2)/(sqrt(2)*dd)
else
 SlH2 := 0;
if HH3 <> -9999 then
 SlH3 := (Hc - HH3)/dd
else
 SlH3 := 0;
if HH4 <> -9999 then
 SlH4 := (Hc - HH4)/(sqrt(2)*dd)
else
 SlH4 := 0;
if HH5 <> -9999 then
 SlH5 := (Hc - HH5)/dd
else
 SlH5 := 0;
if HH6 <> -9999 then
 SlH6 := (Hc - HH6)/(sqrt(2)*dd)
else
 SlH6 := 0;
if HH7 <> -9999 then
 SlH7 := (Hc - HH7)/dd
else
 SlH7 := 0;
if HH8 <> -9999 then
 SlH8 := (Hc - HH8)/(sqrt(2)*dd)
else
 SlH8 := 0;

z[1] := d[1];  // Set z for the first layer. The loop below then sets z for the rest.
// Calculate flows for all soil layers
for j := 1 to 4 do
  begin
   // Layer properties
   if j < 4 then z[j+1] := z[j] + d[j+1];
   if phi[j] <> 0 then
     wfps[j] := Ws[j]/Wsm[j]
   else
     wfps[j] := 0;
   Wsex[j] := 0;

   // Et
   if phi[j] <> 0 then
     begin
      if wfps[j] < thetaw[j]/phi[j] then
       Et[j] := 0
      else if wfps[j] >= thetafc[j]/phi[j] then
       Et[j] := r[j]*Pet
      else
       Et[j] := r[j]*Pet*(1-exp(-bet*Ws[j]/Wsm[j]));
      //       Et[j] := r[j]*Pet*(wfps[j] - thetaw[j]/phi[j])/(thetafc[j]/phi[j] - thetaw[j]/phi[j]);
     end
   else
     Et[j] := 0;

   // Lateral runoff
   if d[j] <> 0 then
     Ksl[j] := Ksl0*(exp(-fl*z[j-1]) - exp(-fl*z[j]))/(d[j]*fl)
   else
     Ksl[j] := 0;
   bl1[j] := exp(6 + 270.562*power(Ksl[j], -0.574));
   bl2[j] := 6 + 144.749*power(Ksl[j], -0.444);
   fq[j] := (1 + exp(-wfps[j]))/(1 + bl1[j]*exp(-bl2[j]*wfps[j])) - 2/(1+bl1[j]);
   QlH1[j] := Ksl[j]*SlH1*fq[j];
   QlH2[j] := Ksl[j]*SlH2*fq[j];
   QlH3[j] := Ksl[j]*SlH3*fq[j];
   QlH4[j] := Ksl[j]*SlH4*fq[j];
   QlH5[j] := Ksl[j]*SlH5*fq[j];
   QlH6[j] := Ksl[j]*SlH6*fq[j];
   QlH7[j] := Ksl[j]*SlH7*fq[j];
   QlH8[j] := Ksl[j]*SlH8*fq[j];
   QlT := QlT + QlH1[j] + QlH2[j] + QlH3[j] + QlH4[j] + QlH5[j] + QlH6[j] + QlH7[j] + QlH8[j];

   // Drainage
   if d[j] <> 0 then
     Ksv[j] := Ksv0*(exp(-fv*z[j-1]) - exp(-fv*z[j]))/(d[j]*fv)
   else
     Ksv[j] := 0;
   bv1[j] := exp(6 + 270.562*power(Ksv[j], -0.574));
   bv2[j] := 6 + 144.749*power(Ksv[j], -0.444);
   fg[j] := (1 + exp(-wfps[j]))/(1 + bv1[j]*exp(-bv2[j]*wfps[j])) - 2/(1+bv1[j]);
   if j < numactivelayer then
      G[j] := Ksv[j]*fg[j]
   else
      G[j] := 0;
  end;

// Surface runoff
QsH1 := Ch*SlH1*Wst;
QsH2 := Ch*SlH2*Wst;
QsH3 := Ch*SlH3*Wst;
QsH4 := Ch*SlH4*Wst;
QsH5 := Ch*SlH5*Wst;
QsH6 := Ch*SlH6*Wst;
QsH7 := Ch*SlH7*Wst;
QsH8 := Ch*SlH8*Wst;

QsT := QsH1 + QsH2 + QsH3 + QsH4 + QsH5 + QsH6 + QsH7 + QsH8;

for j := 1 to 4 do
  dWsdt[j] := Qin[j] + G[j-1] - G[j] - Et[j] - QlH1[j] - QlH2[j] -
      QlH3[j] - QlH4[j] - QlH5[j] - QlH6[j] - QlH7[j] - QlH8[j];
dWsdt[1] := dWsdt[1] + Inf;
if CalculateDiscrete then
begin
// Add any discrete processes here
end; //discrete processes


{ Now calculate the derivatives of the state variables. If the holdConstant
  portion of the state variable is set to true then set the derivative equal to
  zero. }
for jj := 1 to 4 do
begin
 if (tstat[0 + jj].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
  dWsdt[jj] := 0
 else
  dWsdt[jj] := dWsdt[jj];
end;
 
if (tstat[5].HoldConstant) and (FmOptions.RunOptions.HoldStatesConstant) then
 dWstdt := 0
else
 dWstdt := P - Inf - QsH1 - QsH2 - QsH3 - QsH4 - QsH5 - QsH6 - QsH7 - QsH8;
 

{ Now that the calculations are complete, assign everything back into the arrays
  so the rest of the code can access the values calculated here. (Local variables
  are destroyed at the end of the procedure).

  Put the state variables back into the global arrays in case the state variable
  was manually changed in this procedure (e.g. discrete state variables or steady state
  calculations).   }
for jj := 1 to 4 do tstat[0 + jj].value := Ws[jj];
tstat[5].value := Wst;

{  Put all process values into process variable array. The first numstate
  processes are the derivatives of the state variables (Calculated above).}
for jj := 1 to 4 do tproc[0 + jj].value := dWsdt[jj];
tproc[5].value := dWstdt;

{ Now the remaining processes. Be sure to number the processes the same here as
  you did in the procedure counts above. }
for jj := 1 to 4 do tproc[ModelDef.numstate + 0 + jj].value := dum[jj];
tproc[ModelDef.numstate + 5].value := dum2;
for jj := 1 to 4 do tproc[ModelDef.numstate + 5 + jj].value := Et[jj];
tproc[ModelDef.numstate + 10].value := Pet;
tproc[ModelDef.numstate + 11].value := SlH1;
tproc[ModelDef.numstate + 12].value := SlH2;
tproc[ModelDef.numstate + 13].value := SlH3;
tproc[ModelDef.numstate + 14].value := SlH4;
tproc[ModelDef.numstate + 15].value := SlH5;
tproc[ModelDef.numstate + 16].value := SlH6;
tproc[ModelDef.numstate + 17].value := SlH7;
tproc[ModelDef.numstate + 18].value := SlH8;
for jj := 1 to 4 do tproc[ModelDef.numstate + 18 + jj].value := QlH1[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 22 + jj].value := QlH2[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 26 + jj].value := QlH3[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 30 + jj].value := QlH4[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 34 + jj].value := QlH5[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 38 + jj].value := QlH6[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 42 + jj].value := QlH7[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 46 + jj].value := QlH8[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 50 + jj].value := fq[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 54 + jj].value := Ksl[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 58 + jj].value := bl1[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 62 + jj].value := bl2[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 66 + jj].value := G[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 70 + jj].value := fg[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 74 + jj].value := Ksv[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 78 + jj].value := bv1[jj];
for jj := 1 to 4 do tproc[ModelDef.numstate + 82 + jj].value := bv2[jj];
tproc[ModelDef.numstate + 87].value := Inf;
for jj := 1 to 4 do tproc[ModelDef.numstate + 87 + jj].value := Wsex[jj];
tproc[ModelDef.numstate + 92].value := QsH1;
tproc[ModelDef.numstate + 93].value := QsH2;
tproc[ModelDef.numstate + 94].value := QsH3;
tproc[ModelDef.numstate + 95].value := QsH4;
tproc[ModelDef.numstate + 96].value := QsH5;
tproc[ModelDef.numstate + 97].value := QsH6;
tproc[ModelDef.numstate + 98].value := QsH7;
tproc[ModelDef.numstate + 99].value := QsH8;
tproc[ModelDef.numstate + 100].value := QsT;
tproc[ModelDef.numstate + 101].value := QlT;

end;  // End of processes procedure


       { Do not make any modifications to code below this line. }
{****************************************************************************}


{This function counts the parameters in all processes less than processnum.}
function ParCount(processnum:integer) : integer;
var
 NumberofParams, counter : integer;
begin
  NumberofParams := 0;
  for counter := ModelDef.numstate + 1 to processnum - 1 do
         NumberofParams := NumberofParams + proc[counter].parameters;
  ParCount := NumberofParams;
end; // end of parcount function

{ This procedure supplies the derivatives of the state variables to the
  integrator. Since the integrator deals only with the values of the variables
  and not there names, units or the state field HoldConstant, this procedure
  copies the state values into a temporary state array and copies the value of
  HoldConstant into the temporary state array and passes this temporary state
  array to the procedure processes. }
PROCEDURE derivs(t, drt:double; var tdrive:drivearray; var tpar:paramarray;
             var statevalue:yValueArray; VAR dydt:yValueArray);
var
   i:integer;
   tempproc:processarray;
   tempstate:statearray;
begin
   tempstate := stat;  // Copy names, units and HoldConstant to tempstate
  // Copy current values of state variables into tempstate
   for i := 1 to ModelDef.numstate do tempstate[i].value := statevalue[i];
  // Calculate the process values
   processes(t, drt, tdrive, tpar, tempstate, tempproc, false);
  // Put process values into dydt array to get passed back to the integrator.
   for i:= 1 to ModelDef.numstate do dydt[i]:=tempproc[i].value;
end;  // end of derivs procedure

end.
